function ball = makeBall(Nx, Ny, Nz, cx, cy, cz, radius)
%MAKEBALL   Create a binary map of filled ball within a 3D grid.
%
% DESCRIPTION:
%       makeBall creates a binary map of a filled ball within a
%       three-dimensional grid (the ball position is denoted by 1's in the
%       matrix with 0's elsewhere). A single pixel is taken as the disc 
%       centre thus the total diameter of the ball will always be an odd
%       number of pixels. 
%
% USAGE:
%       makeBall(Nx, Ny, Nz, cx, cy, cz, radius)
%
% INPUTS:
%       Nx, Ny, Nz      - size of the 3D grid [number of pixels]
%       cx, cy, cz      - centre of the ball [pixel coordinates]
%       radius          - ball radius [number of pixels]
%
% OUTPUTS:
%       ball            - 3D binary map of a filled ball
%
% ABOUT:
%       author          - Bradley Treeby
%       date            - 1st July 2009
%       last update     - 17th July 2009
%       
% This function is part of the k-Wave Toolbox (http://www.k-wave.org)
% Copyright (C) 2009, 2010 Bradley Treeby and Ben Cox
%
% See also makeCircle, makeDisc, makeSphere 

% This file is part of k-Wave. k-Wave is free software: you can
% redistribute it and/or modify it under the terms of the GNU Lesser
% General Public License as published by the Free Software Foundation,
% either version 3 of the License, or (at your option) any later version.
% 
% k-Wave is distributed in the hope that it will be useful, but WITHOUT ANY
% WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
% FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
% more details. 
% 
% You should have received a copy of the GNU Lesser General Public License
% along with k-Wave. If not, see <http://www.gnu.org/licenses/>.

% define literals
MAGNITUDE = 1;

% create empty matrix
ball = zeros(Nz, Nx, Ny);

% define pixel map
r = makePixelMap(Nx, Ny, Nz, 'Shift', [0 0 0]);

% create disc
ball(r < radius) = MAGNITUDE;

% shift centre
cx = cx - ceil(Nx/2);
cy = cy - ceil(Ny/2);
cz = cz - ceil(Nz/2);
ball = circshift(ball, [cz cx cy]);